/*
 * Decompiled with CFR 0.152.
 */
package com.mapinfo.midev.coordsys.projection;

import com.mapinfo.midev.coordsys.CodeSet;
import com.mapinfo.midev.coordsys.projection.IProjectionFactory;
import com.mapinfo.midev.coordsys.projection.Projection;
import com.mapinfo.midev.coordsys.projection.ProjectionParamType;
import com.mapinfo.midev.coordsys.projection.ProjectionParams;
import com.mapinfo.midev.coordsys.util.CoordSysUtilities;
import com.mapinfo.midev.geometry.DirectPosition;
import com.mapinfo.midev.unit.AngularUnit;
import com.mapinfo.midev.unit.LinearUnit;
import com.mapinfo.midev.util.Code;
import com.mapinfo.midev.util.DoubleRect;
import com.mapinfo.midev.util.MathUtil;
import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

public class TransverseMercatorFactory
implements IProjectionFactory {
    private static final ProjectionParamType[] PARAM_INFO = new ProjectionParamType[]{ProjectionParamType.ORIGIN_LONGITUDE, ProjectionParamType.ORIGIN_LATITUDE, ProjectionParamType.SCALE_FACTOR, ProjectionParamType.FALSE_EASTING, ProjectionParamType.FALSE_NORTHING};
    private static final Set<Code> CODE_SET;
    private final Modification m_modification;

    protected TransverseMercatorFactory(Modification modification) {
        this.m_modification = modification;
    }

    public TransverseMercatorFactory() {
        this(Modification.NONE);
    }

    @Override
    public ProjectionParamType[] getParamInfo() {
        return PARAM_INFO;
    }

    @Override
    public Set<Code> getSupportedCodes() {
        return CODE_SET;
    }

    @Override
    public Projection createProjection(ProjectionParams projectionParams) {
        return new TransverseMercator(new CodeSet(this.getSupportedCodes()), projectionParams, this.m_modification);
    }

    private static double adjlon(double lon) {
        double dTwoPi = Math.PI * 2;
        if (Math.abs(lon) <= Math.PI) {
            return lon;
        }
        lon += Math.PI;
        lon -= dTwoPi * Math.floor(lon / dTwoPi);
        return lon -= Math.PI;
    }

    private static double gatg(double[] p1, double B) {
        double h = 0.0;
        double h1 = p1[p1.length - 1];
        double h2 = 0.0;
        double cos_2B = 2.0 * Math.cos(2.0 * B);
        for (int index = p1.length - 2; index >= 0; --index) {
            h = -h2 + cos_2B * h1 + p1[index];
            h2 = h1;
            h1 = h;
        }
        return B + h * Math.sin(2.0 * B);
    }

    private static double clenS(double[] a, double arg_r, double arg_i, double[] vals) {
        double sin_arg_r = Math.sin(arg_r);
        double cos_arg_r = Math.cos(arg_r);
        double exp_arg_i = Math.exp(arg_i);
        double pxe_arg_i = Math.exp(-arg_i);
        double sinh_arg_i = (exp_arg_i - pxe_arg_i) / 2.0;
        double cosh_arg_i = (exp_arg_i + pxe_arg_i) / 2.0;
        double r = 2.0 * cos_arg_r * cosh_arg_i;
        double i = -2.0 * sin_arg_r * sinh_arg_i;
        double hr = a[a.length - 1];
        double hr1 = 0.0;
        double hi1 = 0.0;
        double hi = 0.0;
        for (int index = a.length - 2; index >= 0; --index) {
            double hr2 = hr1;
            double hi2 = hi1;
            hr1 = hr;
            hi1 = hi;
            hr = -hr2 + r * hr1 - i * hi1 + a[index];
            hi = -hi2 + i * hr1 + r * hi1;
        }
        r = sin_arg_r * cosh_arg_i;
        i = cos_arg_r * sinh_arg_i;
        vals[0] = r * hr - i * hi;
        vals[1] = r * hi + i * hr;
        return vals[0];
    }

    private static double clens(double[] a, double arg_r) {
        double cos_arg_r = Math.cos(arg_r);
        double r = 2.0 * cos_arg_r;
        double hr = a[a.length - 1];
        double hr1 = 0.0;
        for (int index = a.length - 2; index >= 0; --index) {
            double hr2 = hr1;
            hr1 = hr;
            hr = -hr2 + r * hr1 + a[index];
        }
        return Math.sin(arg_r) * hr;
    }

    static {
        HashSet<Code> result = new HashSet<Code>();
        result.add(new Code("epsg", "9807"));
        result.add(new Code("mapinfo", "projection8"));
        CODE_SET = Collections.unmodifiableSet(result);
    }

    protected static enum Modification {
        NONE,
        DENMARK_S34J,
        DENMARK_S34S,
        DENMARK_S45B,
        FINLAND,
        EXTENDED;

    }

    static class TransverseMercator
    extends Projection
    implements Serializable {
        private static final double S34JYMAX = 500000.0;
        private static final double S34JYMIN = -100000.0;
        private static final double S34JXMAX = 500000.0;
        private static final double S34JXMIN = -100000.0;
        private static final double S34SYMAX = 400000.0;
        private static final double S34SYMIN = -100000.0;
        private static final double S34SXMAX = 400000.0;
        private static final double S34SXMIN = -100000.0;
        private static final double S45BYMAX = 150000.0;
        private static final double S45BYMIN = -50000.0;
        private static final double S45BXMAX = 150000.0;
        private static final double S45BXMIN = -50000.0;
        private static final double U32JNMAX = 6500000.0;
        private static final double U32JNMIN = 5900000.0;
        private static final double U32JEMAX = 900000.0;
        private static final double U32JEMIN = 300000.0;
        private static final double U32SNMAX = 6400000.0;
        private static final double U32SNMIN = 5900000.0;
        private static final double U32SEMAX = 1000000.0;
        private static final double U32SEMIN = 400000.0;
        private static final double U33BNMAX = 6200000.0;
        private static final double U33BNMIN = 6000000.0;
        private static final double U33BEMAX = 600000.0;
        private static final double U33BEMIN = 400000.0;
        private static final double[] cForwardS34J = new double[]{11.0, 179608.42195, 1.0001331975, -7.0305752009E-11, -1.5599420745E-17, 1.5445877248E-20, 4.3890319833E-27, -1.6183338075E-30, -5.5641640524E-37, 7.3168052174E-41, -5.5681331488E-48, -1.1898800529E-51, 9.3729805349E-58, -0.019983146896, -7.1626220918E-10, 8.7005607599E-16, 1.2803413015E-20, -1.4192641776E-25, -1.8600522253E-30, 1.9254375816E-35, 9.7422133101E-41, -1.0624197027E-45, -1.8966913523E-51, 1.9980063231E-56, -1.7603869211E-12, 2.5245131678E-16, 1.3597334098E-20, -7.5221602147E-26, -2.7274916388E-30, 1.7704978387E-35, 1.8929469769E-40, -1.004214608E-45, -4.1023552391E-51, 1.8978875611E-56, 5.7374729469E-16, 2.3204950467E-20, 8.7584452002E-27, -4.8904451142E-30, -1.133722334E-35, 4.1563308436E-40, 1.5263628169E-46, -1.0101062278E-50, 9.0844218534E-57, -1.7132615557E-21, -6.2105384175E-26, -1.6823635928E-30, 1.6338290176E-36, 2.115881538E-40, -1.670239383E-45, -7.7415150086E-51, 6.901341003E-56, -2.3212292992E-25, -5.7098757419E-30, 1.5740531826E-35, 2.6141821121E-40, -1.0605463305E-47, -1.2050013481E-50, 2.8385992577E-56, 7.9684988821E-31, 2.7211973029E-35, 8.032475531E-41, -3.6485695942E-46, -4.8480161004E-51, 4.0649973635E-56, 2.7115342914E-35, 7.8845858899E-40, -2.0021883419E-45, -6.1709534864E-51, 3.8404725036E-57, -1.0306208435E-40, -4.0856355905E-45, -1.1672995447E-51, 2.7626939792E-56, -1.0278262874E-45, -3.7985179164E-50, 6.3788054699E-56, 4.7752796291E-51, 2.2043753586E-55, -4.0961492683E-58, 6211357.565339, 259885.74773, -1.0001338073, 3.9172295418E-10, -3.4230073667E-16, -2.2818377692E-20, 2.1440199329E-25, 5.0019389048E-30, -4.2830573831E-35, -4.9751246757E-40, 3.7732561187E-45, 1.8115194532E-50, -1.2659446185E-55, -0.019981958478, 5.210275759E-11, -7.2308108818E-16, -1.5870040358E-20, 3.7855942153E-26, 4.7234232696E-30, -1.2547212518E-35, -5.6120363216E-40, 2.4802558487E-45, 2.4254823739E-50, -1.3399659168E-55, -2.9832623581E-10, 7.9794592713E-16, -8.2061470916E-21, -2.1412572954E-25, 2.2549092808E-30, 1.9760544576E-35, -1.9648841365E-40, -3.5791956901E-46, 8.3762565587E-51, -4.4779684691E-56, 2.8705133213E-16, -1.0202379712E-20, 1.0004204096E-25, 1.5892199642E-30, -7.9348236615E-36, -2.4206841313E-40, 7.3271996448E-46, 1.2265781762E-50, -7.9885986352E-56, -9.4293323888E-21, -6.7854867401E-26, 9.9314808084E-31, 1.3740841353E-35, -1.9398881618E-40, -5.2827833902E-46, 7.4555864194E-51, 1.6454728774E-56, -2.9275805707E-26, 1.6912863842E-30, -1.2123373007E-35, -8.2283656974E-41, 6.7747175215E-46, 8.9516881385E-52, 2.6709952284E-56, 8.217943199E-31, 1.7941344576E-36, -4.8498850776E-41, -1.7705494231E-46, 5.9031266038E-51, -3.5754574387E-56, 1.2785554676E-36, -1.1303931405E-40, 7.2978424008E-46, 3.4247246166E-51, -3.38637577E-56, -3.5127503061E-41, 6.7692928991E-47, 5.5353232299E-52, 4.1057957366E-57, 2.0236690107E-47, 2.4864726958E-51, -1.6699135261E-56, 5.9706206647E-52, -3.8670366463E-57, -1.5348009612E-57, 535887.734971};
        private static final double[] cInverseS34J = new double[]{11.0, 6211357.5431, 0.99946783805, 4.9938687647E-11, 3.7990966203E-17, -1.5339193581E-20, -7.7323064231E-27, 1.592113066E-30, 9.5529708059E-37, -7.1611634315E-41, -1.4830976794E-47, 1.1577689086E-51, -5.7055636656E-58, -0.019969840572, -7.1805559621E-10, 8.4045330539E-16, 1.3682436042E-20, -1.3631677906E-25, -1.9747750527E-30, 1.8282909188E-35, 1.0360374884E-40, -1.0151462821E-45, -2.0151906004E-51, 1.9305008524E-56, 2.3900420211E-11, -2.6486221183E-16, -1.3109796722E-20, 8.8930037465E-26, 2.636745263E-30, -2.0788069675E-35, -1.7881694137E-40, 1.1903122385E-45, 3.8471591094E-51, -2.2681491134E-56, 5.8496451149E-16, 2.4094520047E-20, 1.2069686697E-26, -4.9977313546E-30, -9.6921660473E-36, 4.2026524388E-40, 1.3405876318E-46, -1.0131754273E-50, 7.27400189E-57, 7.8465984626E-22, 3.8634193994E-26, 1.4338062213E-30, 4.363821309E-37, -2.2938670833E-40, 1.6535643482E-45, 8.026326111E-51, -6.7548306357E-56, -2.3686862535E-25, -5.9285339942E-30, 1.2257380281E-35, 2.7868104605E-40, -1.3509809193E-46, -1.2352581563E-50, 3.3545921272E-56, -5.7946172707E-31, -2.4002504495E-35, 1.2383768529E-41, 1.4988400999E-46, 5.2572256175E-51, -4.3211999874E-56, 2.8395578205E-35, 8.1206547484E-40, -1.355673804E-45, -7.1034554654E-51, 3.6760430426E-57, 7.6932686862E-41, 3.9826299606E-45, -5.2280161357E-51, -2.1625606715E-56, -1.1790660973E-45, -3.9043014229E-50, 2.1421469282E-56, -3.632880943E-51, -2.2294273001E-55, 6.4954545832E-57, 179608.367287, 535889.35583, -0.99946722875, 3.9169429886E-10, 3.4406219619E-16, -2.3100209513E-20, -2.0980875706E-25, 5.0916401948E-30, 4.2352645285E-35, -5.0814423882E-40, -3.7815509102E-45, 1.8578734513E-50, 1.2844016085E-55, -0.019968652797, -1.0130504008E-11, -6.6347235578E-16, 1.38551445E-20, 7.2141514059E-27, -4.0737530727E-30, -5.2561571812E-36, 4.6984096369E-40, 1.6930479011E-45, -2.0002012019E-50, -1.0245050164E-55, -2.998493237E-10, -8.5765805237E-16, -7.6496969563E-21, 2.090910339E-25, 1.8677723038E-30, -2.054658711E-35, -1.3700082509E-40, 6.7985231879E-46, 5.1001963923E-51, 2.4562931015E-56, 2.6953902564E-16, 1.033509851E-20, 1.0512237646E-25, -1.4032065963E-30, -8.8653883615E-36, 2.3077430247E-40, 7.5592438487E-46, -1.1543418844E-50, -7.1008824555E-56, -8.9014771332E-21, 7.7463136654E-26, 1.0174620943E-30, -1.294229525E-35, -1.7285851072E-40, 5.4866318479E-46, 5.7374154072E-51, -3.0826966859E-56, -2.763371753E-26, -1.7031103658E-30, -1.2230749083E-35, 6.5347121093E-41, 6.7871460868E-46, -5.8126135877E-52, 1.9440416443E-56, 7.5394910385E-31, -2.8282134462E-36, -5.6124165669E-41, 1.2596878394E-46, 6.0931461255E-51, 4.212326469E-56, 1.2127899226E-36, 1.1389363448E-40, 7.2153811444E-46, -2.8345320964E-51, -2.7877471524E-56, -3.1312337444E-41, -2.1103067283E-47, 7.3409941617E-52, -3.6764625883E-57, 1.942449899E-47, -2.5238035439E-51, -1.7142319035E-56, 5.2147178289E-52, 3.106741796E-57, -1.4447927077E-57, 259884.1271};
        private static final double[] cForwardS34S = new double[]{11.0, 108580.1864, 0.99979577051, -1.7628943162E-11, 1.0427534895E-15, 1.1752660013E-20, -4.1725951789E-25, -2.0886013352E-30, 5.0159910521E-35, 8.1544720099E-41, -1.9858454477E-45, 1.0287586664E-51, 1.5240493624E-56, -0.019606882974, -4.3334445849E-9, -8.3389760421E-16, 1.0973370628E-20, 4.16797142E-25, -7.6223106762E-30, -2.6974169389E-35, 1.0559766833E-39, -7.9105362819E-46, -4.3221579077E-50, 1.7386643471E-55, 1.2383447417E-11, 2.4562547942E-15, 6.8925282176E-21, -1.7152848763E-24, -7.9695044091E-30, 3.3130334355E-34, 7.2753870864E-40, -2.6300617494E-44, 1.8761736758E-50, 4.9342284156E-55, -1.188331636E-15, 3.0432347639E-20, 8.7215487883E-25, -1.250443026E-29, -1.2227674272E-34, 2.9081999906E-39, -3.1952240217E-44, -1.4743224736E-49, 1.2568548821E-54, -2.5071180419E-20, -5.9524296653E-25, 9.2424060664E-31, 4.9573493239E-34, 2.2448369232E-39, -7.0754967066E-45, -5.6354749924E-49, 2.1781478699E-54, 2.2253781457E-26, -6.2890039241E-30, -2.0008683312E-34, -1.7020460684E-39, 3.2555843309E-44, 9.8134258366E-49, -2.4483421139E-54, 1.8839740115E-29, -2.3538560402E-34, 2.2376842442E-39, -9.7171211402E-44, -1.2247888683E-48, 9.698775351E-54, 1.7284820253E-34, -1.1091432215E-39, 2.7533441703E-44, 7.8319181256E-49, -1.4799366349E-53, -3.0018484236E-39, 6.9348411599E-44, -4.1616647911E-49, 1.3152283302E-53, -3.3804637596E-44, 1.9935807263E-49, -2.6561161114E-54, 1.4397002066E-49, -4.71713465E-54, 1.7956467135E-54, 6143171.658806, 118032.75921, -0.99979562993, 2.2096993755E-9, -7.0235014619E-16, -2.5468665317E-20, 2.372138019E-25, 4.7914394273E-30, -2.9235335621E-35, -3.1752558266E-40, 1.4296733762E-45, 5.6318539715E-51, -2.1372653209E-56, -0.019608396525, 2.693007684E-11, -1.2459885445E-15, -5.4286069171E-20, 2.9240791919E-26, 3.9538770772E-29, 2.7013028627E-34, -7.5406070323E-39, -5.985674659E-44, 4.5446492893E-49, 3.45222563E-54, -2.1613106297E-9, -1.0284937475E-15, -4.752486802E-21, 6.3011943756E-25, -2.1972028767E-29, -5.0003754225E-34, 9.9050910142E-39, 1.3059721524E-43, -8.7855986019E-49, -9.5972480432E-54, 6.5442913745E-16, -3.5437986694E-21, 9.6702070364E-25, 2.6540324328E-29, -2.7203478976E-34, -1.3876118299E-38, -6.840507998E-44, 1.3551063996E-48, 1.2134346241E-53, 3.6698571394E-21, 3.5038598575E-25, 2.5366408419E-30, 1.7484199561E-34, 6.0455102679E-39, 4.0847580091E-44, -1.1488873067E-48, -1.1846835071E-53, -2.0809106978E-25, -1.0226779365E-31, -2.9096633166E-34, -4.6363312354E-39, 5.9140904027E-44, 1.2923519124E-48, 3.8990462602E-54, -1.7976932688E-30, -3.5615208413E-35, 6.5231103343E-40, -3.1233789979E-44, -7.9467366298E-49, -4.5651670959E-54, 2.9008640902E-35, 1.1442713527E-40, 2.5264390889E-44, 4.3770693863E-49, 2.3499600661E-54, 1.3323328179E-40, 1.492955877E-45, -1.1292325082E-49, -1.2081269452E-54, -1.5429064621E-45, -5.2570211474E-51, -2.0928319154E-55, -3.8450652804E-52, -1.8046153053E-56, 1.3843491805E-56, 679082.320128};
        private static final double[] cInverseS34S = new double[]{11.0, 6143169.5373, 0.99981973699, -1.0964736421E-10, -1.0559048632E-15, -1.1478094008E-20, 4.2172459636E-25, 1.9084615741E-30, -5.0227991843E-35, -5.8508147485E-41, 1.9497558465E-45, -1.9030604163E-51, -1.1674992376E-56, -0.019607346629, -4.3277818384E-9, -8.4535742476E-16, 1.1669207355E-20, 4.358833449E-25, -7.5235775082E-30, -3.2448901159E-35, 1.0341823305E-39, -1.6467088951E-46, -4.3465306092E-50, 1.5800702291E-55, 1.1590764287E-10, -2.4877495594E-15, -5.8449369708E-21, 1.7495080863E-24, 8.0567279658E-30, -3.4119954537E-34, -7.0441405933E-40, 2.5040727605E-44, -1.9753554317E-50, -4.6623953673E-55, -1.1284237816E-15, 3.372630941E-20, 8.0048550624E-25, -1.3724675347E-29, -1.3146204233E-34, 2.8769060513E-39, -3.3903070242E-44, -1.0381481775E-49, 1.1739253553E-54, 2.3954304515E-20, 5.6451650666E-25, -1.3680967238E-30, -5.0629522861E-34, -2.5787791044E-39, 1.5749534013E-44, 6.6588168655E-49, -2.6117602308E-54, 8.226676956E-27, -9.2964661316E-30, -1.2939153428E-34, -1.4844961079E-39, 4.1637777164E-44, 1.0303914205E-48, -3.2921479391E-54, -1.8613629122E-29, 2.7083908133E-34, -2.0389539621E-39, 9.4913410692E-44, 1.2105180969E-48, -1.1393091754E-53, 1.6802827184E-34, -3.8788506077E-40, 7.4546018646E-45, 7.242661059E-49, -1.5499544059E-53, 3.015195768E-39, -7.7126572795E-44, 3.8393508564E-49, -1.1997717651E-53, -3.2368554928E-44, 1.4494519184E-49, -7.2998087473E-55, -1.475496532E-49, 5.2320585872E-54, 1.6983633886E-54, 108578.008582, 679085.21383, -0.99981987793, 2.2067947668E-9, 7.4080643569E-16, -2.7024780096E-20, -2.3860242186E-25, 5.9362108595E-30, 2.0903490479E-35, -5.2324223549E-40, 3.2546718002E-46, 1.7246428336E-50, -7.6058409579E-56, -0.019608860322, 2.2939412486E-10, -1.1975947428E-15, 5.1738651841E-20, 2.197469046E-26, -3.7854430635E-29, 2.4792584222E-34, 7.09602938E-39, -5.3320738562E-44, -4.2095102312E-49, 2.9767318072E-54, -2.1585027913E-9, 9.8525890561E-16, -1.7197758382E-21, -6.9990147248E-25, -2.4134130168E-29, 5.4768040973E-34, 1.0114168023E-38, -1.3537852094E-43, -8.8213877335E-49, 9.527178641E-54, 6.9534612921E-16, 2.8580028058E-21, 9.2085175117E-25, -2.8226680106E-29, -1.9847960021E-34, 1.4579803275E-38, -8.4649708831E-44, -1.4155222651E-48, 1.3068571653E-53, 3.9667018491E-21, -3.0022877227E-25, 7.615510597E-31, -1.6251982395E-34, 7.0189181438E-39, -5.5846585229E-44, -1.2341266183E-48, 1.3669455963E-53, -2.2275960734E-25, 5.4974771911E-31, -2.9807621033E-34, 4.9962627827E-39, 5.0563831711E-44, -1.3546107308E-48, 5.2970036586E-54, -1.8327143317E-30, 2.0490556946E-35, 9.5949163898E-40, 3.2682015932E-44, -8.7328117126E-49, 4.8115902425E-54, 3.0563195217E-35, -1.2820482439E-40, 2.6835215111E-44, -4.8061624868E-49, 2.645800673E-54, 1.3244590931E-40, -1.72165476E-46, -1.3967859793E-49, 1.408393111E-54, -1.5999084278E-45, 1.4643697408E-51, -1.2856212972E-55, -2.9347838078E-52, 3.9852449357E-57, 1.4360336479E-56, 118029.907698};
        private static final double[] cForwardS45B = new double[]{4.0, 52222.355954, 1.0003999918, -1.9543132669E-10, 1.9214004196E-15, -1.2249813217E-19, 0.0014741055574, 7.9465429957E-10, -2.5561175816E-14, -1.5415582961E-20, 2.2187819566E-10, -2.2425885615E-14, 1.8415412E-18, -1.0926417356E-14, 6.5851285007E-19, -1.2766025333E-18, 6109645.68068, 47875.457674, -1.0004002847, -5.5739930409E-10, -3.7954137372E-15, -8.5294692035E-19, 0.0014721003307, 3.703301941E-10, -4.377332458E-15, -4.8011997156E-18, 5.7044426679E-10, 9.4892208043E-15, 1.1706710507E-18, 4.2324194994E-15, 1.6689060255E-18, -1.1260303216E-19, 495134.215516};
        private static final double[] cInverseS45B = new double[]{4.0, 6109645.6608, 0.99959800064, 1.9318586062E-10, -1.8870172378E-15, 1.2245658657E-19, 0.0014729233533, 7.9546531725E-10, -2.5580188922E-14, -6.8693684196E-21, -2.1962553713E-10, 2.2369816885E-14, -1.8424778852E-18, -1.0881273228E-14, 6.3729275085E-19, 1.2762267341E-18, 52222.336107, 495134.23016, -0.99959770795, -5.575816853E-10, 3.7654916286E-15, -8.4235346472E-19, 0.0014709197601, -3.6540821608E-10, -4.380537147E-15, 4.8008973854E-18, 5.7059554739E-10, -9.4793524523E-15, 1.1370338222E-18, 4.2365300238E-15, -1.6695314444E-18, -1.0981554453E-19, 47875.442993};
        private final Modification m_modification;
        private double m_eprime2;
        private double m_lambda0;
        private double m_k0;
        private double m_M0;
        private double m_MultY;
        private double m_Aprime;
        private double m_Bprime;
        private double m_Cprime;
        private double m_Aprime2;
        private double m_Bprime2;
        private double m_Cprime2;
        private double m_Dprime2;
        private DoubleRect m_UTMClipRect;
        private DoubleRect m_DanishClipRect;
        private double[] m_pCoeffUTM2Danish;
        private double[] m_pCoeffDanish2UTM;
        private boolean m_bDanish;
        private double m_Qn;
        private double m_Zb;
        private double[] m_cgb = new double[5];
        private double[] m_cbg = new double[5];
        private double[] m_utg = new double[5];
        private double[] m_gtu = new double[5];

        public TransverseMercator(CodeSet codeSet, ProjectionParams projectionParams, Modification modification) {
            super(TransverseMercator.getDisplayNameWithModification(modification), codeSet, projectionParams);
            this.m_modification = modification;
            this.calcInternals();
        }

        private static String getDisplayNameWithModification(Modification modification) {
            String result = "Transverse Mercator";
            switch (modification) {
                case DENMARK_S34J: {
                    result = result + " modified for Danish System 34 Jylland-Fyn";
                    break;
                }
                case DENMARK_S34S: {
                    result = result + " modified for Danish System 34 Sjaelland";
                    break;
                }
                case DENMARK_S45B: {
                    result = result + " modified for Danish System 34 / 45 Bornholm";
                    break;
                }
                case FINLAND: {
                    result = result + " modified for Finnish KKJ";
                    break;
                }
                case EXTENDED: {
                    result = result + " (Extended)";
                }
            }
            return result;
        }

        private void calcInternals() {
            ProjectionParams projectionParams = this.getProjectionParams();
            this.m_eprime2 = projectionParams.getEllipsoid().getEccentricitySquared() / (1.0 - projectionParams.getEllipsoid().getEccentricitySquared());
            this.m_lambda0 = CoordSysUtilities.convertClip(projectionParams.getOriginLongitude(), AngularUnit.RADIAN, -360.0, 360.0);
            double phi0 = CoordSysUtilities.convertClip(projectionParams.getOriginLatitude(), AngularUnit.RADIAN, -90.0, 90.0);
            this.m_k0 = projectionParams.getScaleFactor();
            if (this.m_k0 == 0.0) {
                this.m_k0 = 1.0;
            }
            double e4 = projectionParams.getEllipsoid().getEccentricitySquared() * projectionParams.getEllipsoid().getEccentricitySquared();
            double e6 = projectionParams.getEllipsoid().getEccentricitySquared() * e4;
            this.m_MultY = projectionParams.getEllipsoid().getSemiMajorAxis() * (1.0 - projectionParams.getEllipsoid().getEccentricitySquared() / 4.0 - 3.0 * e4 / 64.0 - 5.0 * e6 / 256.0);
            double A = projectionParams.getEllipsoid().getSemiMajorAxis() * -(3.0 * projectionParams.getEllipsoid().getEccentricitySquared() / 8.0 + 3.0 * e4 / 32.0 + 45.0 * e6 / 1024.0);
            double B = projectionParams.getEllipsoid().getSemiMajorAxis() * (15.0 * e4 / 256.0 + 45.0 * e6 / 1024.0);
            double C = projectionParams.getEllipsoid().getSemiMajorAxis() * -35.0 * e6 / 3072.0;
            this.m_Aprime = A - C;
            this.m_Bprime = 2.0 * B;
            this.m_Cprime = 4.0 * C;
            double phi2 = phi0 * 2.0;
            double cosphi2 = Math.cos(phi2);
            this.m_M0 = this.m_MultY * phi0 + Math.sin(phi2) * (this.m_Aprime + cosphi2 * (this.m_Bprime + cosphi2 * this.m_Cprime));
            double temp = Math.sqrt(1.0 - projectionParams.getEllipsoid().getEccentricitySquared());
            double e1 = (1.0 - temp) / (1.0 + temp);
            double e12 = e1 * e1;
            double e13 = e12 * e1;
            double e14 = e12 * e12;
            A = 3.0 * e1 / 2.0 - 27.0 * e13 / 32.0;
            B = 21.0 * e12 / 16.0 - 55.0 * e14 / 32.0;
            C = 151.0 * e13 / 96.0;
            double D = 1097.0 * e14 / 512.0;
            this.m_Aprime2 = A - C;
            this.m_Bprime2 = 2.0 * B - 4.0 * D;
            this.m_Cprime2 = 4.0 * C;
            this.m_Dprime2 = 8.0 * D;
            switch (this.m_modification) {
                case DENMARK_S34J: {
                    this.m_UTMClipRect = new DoubleRect(300000.0, 5900000.0, 900000.0, 6500000.0);
                    this.m_DanishClipRect = new DoubleRect(-100000.0, -100000.0, 500000.0, 500000.0);
                    this.m_pCoeffUTM2Danish = cForwardS34J;
                    this.m_pCoeffDanish2UTM = cInverseS34J;
                    this.m_bDanish = true;
                    break;
                }
                case DENMARK_S34S: {
                    this.m_UTMClipRect = new DoubleRect(400000.0, 5900000.0, 1000000.0, 6400000.0);
                    this.m_DanishClipRect = new DoubleRect(-100000.0, -100000.0, 400000.0, 400000.0);
                    this.m_pCoeffUTM2Danish = cForwardS34S;
                    this.m_pCoeffDanish2UTM = cInverseS34S;
                    this.m_bDanish = true;
                    break;
                }
                case DENMARK_S45B: {
                    this.m_UTMClipRect = new DoubleRect(400000.0, 6000000.0, 600000.0, 6200000.0);
                    this.m_DanishClipRect = new DoubleRect(-50000.0, -50000.0, 150000.0, 150000.0);
                    this.m_pCoeffUTM2Danish = cForwardS45B;
                    this.m_pCoeffDanish2UTM = cInverseS45B;
                    this.m_bDanish = true;
                    break;
                }
                case EXTENDED: {
                    double n;
                    double lat0 = projectionParams.getOriginLatitude().getValue(AngularUnit.DEGREE) * Math.PI / 180.0;
                    double f = projectionParams.getEllipsoid().getFlattening();
                    double np = n = f / (2.0 - f);
                    this.m_cgb[0] = n * (2.0 + n * (-0.6666666666666666 + n * (-2.0 + n * (2.577777777777778 + n * 0.5777777777777777))));
                    this.m_cbg[0] = n * (-2.0 + n * (0.6666666666666666 + n * (1.3333333333333333 + n * (-1.8222222222222222 + n * 0.7111111111111111))));
                    this.m_cgb[1] = (np *= n) * (2.3333333333333335 + n * (-1.6 + n * (-5.044444444444444 + n * 8.584126984126984)));
                    this.m_cbg[1] = np * (1.6666666666666667 + n * (-1.0666666666666667 + n * (-1.4444444444444444 + n * 2.86984126984127)));
                    this.m_cgb[2] = (np *= n) * (3.7333333333333334 + n * (-3.8857142857142857 + n * 12.019047619047619));
                    this.m_cbg[2] = np * (-1.7333333333333334 + n * (1.619047619047619 + n * 1.6));
                    this.m_cgb[3] = (np *= n) * (6.792063492063492 + n * -9.2);
                    this.m_cbg[3] = np * (1.9634920634920634 + n * -2.4);
                    this.m_cgb[4] = (np *= n) * 13.250793650793652;
                    this.m_cbg[4] = np * -2.3301587301587303;
                    np = n * n;
                    this.m_Qn = this.m_k0 / (1.0 + n) * (1.0 + np * (0.25 + np * (0.015625 + np / 256.0)));
                    this.m_utg[0] = n * (-0.5 + n * (0.6666666666666666 + n * (-0.3854166666666667 + n * (0.002777777777777778 + n * 0.158203125))));
                    this.m_gtu[0] = n * (0.5 + n * (-0.6666666666666666 + n * (0.3125 + n * (0.22777777777777777 + n * -0.4409722222222222))));
                    this.m_utg[1] = np * (-0.020833333333333332 + n * (-0.06666666666666667 + n * (0.3034722222222222 + n * -0.4380952380952381)));
                    this.m_gtu[1] = np * (0.2708333333333333 + n * (-0.6 + n * (0.38680555555555557 + n * 0.44603174603174606)));
                    this.m_utg[2] = (np *= n) * (-0.035416666666666666 + n * (0.04404761904761905 + n * 0.046651785714285715));
                    this.m_gtu[2] = np * (0.25416666666666665 + n * (-0.7357142857142858 + n * 0.5603050595238095));
                    this.m_utg[3] = (np *= n) * (-0.02726314484126984 + n * 0.021825396825396824);
                    this.m_gtu[3] = np * (0.30729786706349205 + n * -1.0654761904761905);
                    this.m_utg[4] = (np *= n) * -0.02841641865079365;
                    this.m_gtu[4] = np * 0.4306671626984127;
                    double Z = TransverseMercatorFactory.gatg(this.m_cbg, lat0);
                    this.m_Zb = -this.m_Qn * (Z + TransverseMercatorFactory.clens(this.m_gtu, 2.0 * Z));
                }
            }
        }

        @Override
        public DirectPosition fromLongLat(DirectPosition source, DirectPosition destination) {
            ProjectionParams projectionParams = this.getProjectionParams();
            double x = CoordSysUtilities.convertClip(source.getX(), AngularUnit.DEGREE, AngularUnit.RADIAN, -360.0, 360.0);
            double y = CoordSysUtilities.convertClip(source.getY(), AngularUnit.DEGREE, AngularUnit.RADIAN, -90.0, 90.0);
            switch (this.m_modification) {
                case FINLAND: {
                    double n = projectionParams.getEllipsoid().getFlattening() / (2.0 - projectionParams.getEllipsoid().getFlattening());
                    double A_KKJ = projectionParams.getEllipsoid().getSemiMajorAxis() * (1.0 + n * n / 4.0 + n * n * n * n / 64.0) / (1.0 + n);
                    double n2 = n * n;
                    double n3 = n2 * n;
                    double n4 = n3 * n;
                    x -= this.m_lambda0;
                    double tmp = Math.tan(y);
                    double Q1 = Math.log(tmp + Math.sqrt(tmp * tmp + 1.0));
                    tmp = projectionParams.getEllipsoid().getEccentricity() * Math.sin(y);
                    double Q2 = 0.5 * Math.log((1.0 + tmp) / (1.0 - tmp));
                    double Q = Q1 - projectionParams.getEllipsoid().getEccentricity() * Q2;
                    double l = x;
                    double h1 = n / 2.0 - 2.0 * n2 / 3.0 + 5.0 * n3 / 16.0 + 41.0 * n4 / 180.0;
                    double h2 = 13.0 * n2 / 48.0 - 3.0 * n3 / 5.0 + 557.0 * n4 / 1440.0;
                    double h3 = 61.0 * n3 / 240.0 - 103.0 * n4 / 140.0;
                    double h4 = 49561.0 * n4 / 161280.0;
                    tmp = (Math.exp(Q) - Math.exp(-Q)) / 2.0;
                    double beta = Math.atan(tmp);
                    tmp = Math.cos(beta) * Math.sin(l);
                    double eta1 = 0.5 * Math.log((1.0 + tmp) / (1.0 - tmp));
                    tmp = 2.0 / (Math.exp(eta1) + Math.exp(-eta1));
                    double ksi1 = Math.asin(MathUtil.clip((double)(Math.sin(beta) / tmp), (double)-1.0, (double)1.0));
                    double ksi = ksi1 + h1 * Math.sin(2.0 * ksi1) * (Math.exp(2.0 * eta1) + Math.exp(-2.0 * eta1)) / 2.0 + h2 * Math.sin(4.0 * ksi1) * (Math.exp(4.0 * eta1) + Math.exp(-4.0 * eta1)) / 2.0 + h3 * Math.sin(6.0 * ksi1) * (Math.exp(6.0 * eta1) + Math.exp(-6.0 * eta1)) / 2.0 + h4 * Math.sin(8.0 * ksi1) * (Math.exp(8.0 * eta1) + Math.exp(-8.0 * eta1)) / 2.0;
                    double eta = eta1 + h1 * Math.cos(2.0 * ksi1) * (Math.exp(2.0 * eta1) - Math.exp(-2.0 * eta1)) / 2.0 + h2 * Math.cos(4.0 * ksi1) * (Math.exp(4.0 * eta1) - Math.exp(-4.0 * eta1)) / 2.0 + h3 * Math.cos(6.0 * ksi1) * (Math.exp(6.0 * eta1) - Math.exp(-6.0 * eta1)) / 2.0 + h4 * Math.cos(8.0 * ksi1) * (Math.exp(8.0 * eta1) - Math.exp(-8.0 * eta1)) / 2.0;
                    y = A_KKJ * ksi * this.m_k0;
                    x = A_KKJ * eta * this.m_k0;
                    x = LinearUnit.convert((double)x, (LinearUnit)projectionParams.getEllipsoid().getAxisUnit(), (LinearUnit)projectionParams.getLinearUnit()) + projectionParams.getFalseEasting().getValue(projectionParams.getLinearUnit());
                    y = LinearUnit.convert((double)y, (LinearUnit)projectionParams.getEllipsoid().getAxisUnit(), (LinearUnit)projectionParams.getLinearUnit()) + projectionParams.getFalseNorthing().getValue(projectionParams.getLinearUnit());
                    break;
                }
                case EXTENDED: {
                    double a = projectionParams.getEllipsoid().getSemiMajorAxis();
                    double lon0 = projectionParams.getOriginLongitude().getValue(AngularUnit.DEGREE) * Math.PI / 180.0;
                    double dHalfPi = 1.5707963267948966;
                    double dMinLongitude = lon0 - 1.5707963267948966;
                    double dMaxLongitude = lon0 + 1.5707963267948966;
                    if (x <= dMinLongitude) {
                        x = dMinLongitude + 0.01;
                    } else if (x >= dMaxLongitude) {
                        x = dMaxLongitude - 0.01;
                    }
                    double t = Math.abs(y) - dHalfPi;
                    if (Math.abs(t) <= 1.0E-12) {
                        y = y < 0.0 ? -dHalfPi : dHalfPi;
                    }
                    x -= lon0;
                    x = TransverseMercatorFactory.adjlon(x);
                    double Cn = y;
                    double Ce = x;
                    Cn = TransverseMercatorFactory.gatg(this.m_cbg, Cn);
                    double sin_Cn = Math.sin(Cn);
                    double cos_Cn = Math.cos(Cn);
                    double sin_Ce = Math.sin(Ce);
                    double cos_Ce = Math.cos(Ce);
                    Cn = Math.atan2(sin_Cn, cos_Ce * cos_Cn);
                    Ce = Math.atan2(sin_Ce * cos_Cn, Math.hypot(sin_Cn, cos_Cn * cos_Ce));
                    Ce = Math.log(Math.tan(0.7853981633974483 + Ce * 0.5));
                    double[] vals = new double[2];
                    Cn += TransverseMercatorFactory.clenS(this.m_gtu, 2.0 * Cn, 2.0 * Ce, vals);
                    if (Math.abs(Ce += vals[1]) <= 2.623395162778) {
                        y = (this.m_Qn * Cn + this.m_Zb) * a;
                        x = this.m_Qn * Ce * a;
                    } else {
                        y = 0.0;
                        x = 0.0;
                    }
                    x = LinearUnit.convert((double)x, (LinearUnit)projectionParams.getEllipsoid().getAxisUnit(), (LinearUnit)projectionParams.getLinearUnit()) + projectionParams.getFalseEasting().getValue(projectionParams.getLinearUnit());
                    y = LinearUnit.convert((double)y, (LinearUnit)projectionParams.getEllipsoid().getAxisUnit(), (LinearUnit)projectionParams.getLinearUnit()) + projectionParams.getFalseNorthing().getValue(projectionParams.getLinearUnit());
                    break;
                }
                default: {
                    double phi = y;
                    double phi2 = phi * 2.0;
                    double cosphi2 = Math.cos(phi2);
                    double M = this.m_MultY * phi + Math.sin(phi2) * (this.m_Aprime + cosphi2 * (this.m_Bprime + cosphi2 * this.m_Cprime));
                    double cosphi = Math.cos(phi);
                    if (cosphi == 0.0) {
                        x = 0.0;
                        y = this.m_k0 * (M - this.m_M0);
                    } else {
                        double sinphi = Math.sin(phi);
                        double N = projectionParams.getEllipsoid().getSemiMajorAxis() / Math.sqrt(1.0 - projectionParams.getEllipsoid().getEccentricitySquared() * sinphi * sinphi);
                        double tanphi = Math.tan(phi);
                        double T = tanphi * tanphi;
                        double T2 = T * T;
                        double C = this.m_eprime2 * cosphi * cosphi;
                        double C2 = C * C;
                        double A = (x - this.m_lambda0) * cosphi;
                        double A2 = A * A;
                        double A3 = A * A2;
                        double A4 = A2 * A2;
                        double A5 = A2 * A3;
                        double A6 = A3 * A3;
                        x = this.m_k0 * N * (A + (1.0 - T + C) * A3 / 6.0 + (5.0 - 18.0 * T + T2 + 72.0 * C - 58.0 * this.m_eprime2) * A5 / 120.0);
                        y = this.m_k0 * (M - this.m_M0 + N * tanphi * (A2 / 2.0 + (5.0 - T + 9.0 * C + 4.0 * C2) * A4 / 24.0 + (61.0 - 58.0 * T + T2 + 600.0 * C - 330.0 * this.m_eprime2) * A6 / 720.0));
                    }
                    x = LinearUnit.convert((double)x, (LinearUnit)projectionParams.getEllipsoid().getAxisUnit(), (LinearUnit)projectionParams.getLinearUnit()) + projectionParams.getFalseEasting().getValue(projectionParams.getLinearUnit());
                    y = LinearUnit.convert((double)y, (LinearUnit)projectionParams.getEllipsoid().getAxisUnit(), (LinearUnit)projectionParams.getLinearUnit()) + projectionParams.getFalseNorthing().getValue(projectionParams.getLinearUnit());
                    if (!this.m_bDanish) break;
                    DirectPosition tmp = TransverseMercator.clipPointToRect(this.m_UTMClipRect, x, y);
                    tmp = this.ConvertDanish(tmp, DanishDirection.ToDanish);
                    x = -tmp.getX();
                    y = tmp.getY();
                }
            }
            destination.setXY(x, y);
            return destination;
        }

        @Override
        public DirectPosition toLongLat(DirectPosition source, DirectPosition destination) {
            ProjectionParams projectionParams = this.getProjectionParams();
            DirectPosition tmpDP = new DirectPosition(source);
            if (this.m_bDanish) {
                tmpDP.setXY(-tmpDP.getX(), tmpDP.getY());
                tmpDP = TransverseMercator.clipPointToRect(this.m_DanishClipRect, tmpDP.getX(), tmpDP.getY());
                tmpDP = this.ConvertDanish(tmpDP, DanishDirection.ToUTM);
            }
            double x = tmpDP.getX() - projectionParams.getFalseEasting().getValue(projectionParams.getLinearUnit());
            double y = tmpDP.getY() - projectionParams.getFalseNorthing().getValue(projectionParams.getLinearUnit());
            x = LinearUnit.convert((double)x, (LinearUnit)projectionParams.getLinearUnit(), (LinearUnit)projectionParams.getEllipsoid().getAxisUnit());
            y = LinearUnit.convert((double)y, (LinearUnit)projectionParams.getLinearUnit(), (LinearUnit)projectionParams.getEllipsoid().getAxisUnit());
            y = MathUtil.clip((double)y, (double)-1.0E9, (double)1.0E9);
            switch (this.m_modification) {
                case FINLAND: {
                    double Q;
                    double n = projectionParams.getEllipsoid().getFlattening() / (2.0 - projectionParams.getEllipsoid().getFlattening());
                    double A_KKJ = projectionParams.getEllipsoid().getSemiMajorAxis() * (1.0 + n * n / 4.0 + n * n * n * n / 64.0) / (1.0 + n);
                    double e = Math.sqrt(projectionParams.getEllipsoid().getEccentricitySquared());
                    double n2 = n * n;
                    double n3 = n2 * n;
                    double n4 = n3 * n;
                    double ksi = y / A_KKJ / this.m_k0;
                    double eta = x / A_KKJ / this.m_k0;
                    double h1 = n / 2.0 - 2.0 * n2 / 3.0 + 37.0 * n3 / 96.0 - n4 / 360.0;
                    double h2 = n2 / 48.0 + n3 / 45.0 - 437.0 * n4 / 1440.0;
                    double h3 = 17.0 * n3 / 480.0 - 37.0 * n4 / 840.0;
                    double h4 = 4397.0 * n4 / 161280.0;
                    double ksi1 = ksi - h1 * Math.sin(2.0 * ksi) * (Math.exp(2.0 * eta) + Math.exp(-2.0 * eta)) / 2.0 - h2 * Math.sin(4.0 * ksi) * (Math.exp(4.0 * eta) + Math.exp(-4.0 * eta)) / 2.0 - h3 * Math.sin(6.0 * ksi) * (Math.exp(6.0 * eta) + Math.exp(-6.0 * eta)) / 2.0 - h4 * Math.sin(8.0 * ksi) * (Math.exp(8.0 * eta) + Math.exp(-8.0 * eta)) / 2.0;
                    double eta1 = eta - h1 * Math.cos(2.0 * ksi) * (Math.exp(2.0 * eta) - Math.exp(-2.0 * eta)) / 2.0 - h2 * Math.cos(4.0 * ksi) * (Math.exp(4.0 * eta) - Math.exp(-4.0 * eta)) / 2.0 - h3 * Math.cos(6.0 * ksi) * (Math.exp(6.0 * eta) - Math.exp(-6.0 * eta)) / 2.0 - h4 * Math.cos(8.0 * ksi) * (Math.exp(8.0 * eta) - Math.exp(-8.0 * eta)) / 2.0;
                    double tmp = 2.0 / (Math.exp(eta1) + Math.exp(-eta1));
                    double beta = Math.asin(MathUtil.clip((double)(tmp * Math.sin(ksi1)), (double)-1.0, (double)1.0));
                    tmp = (Math.exp(eta1) - Math.exp(-eta1)) / (Math.exp(eta1) + Math.exp(-eta1));
                    double l = Math.asin(MathUtil.clip((double)(tmp / Math.cos(beta)), (double)-1.0, (double)1.0));
                    tmp = Math.tan(beta);
                    double Q1 = Q = Math.log(tmp + Math.sqrt(tmp * tmp + 1.0));
                    for (int i = 0; i < 10; ++i) {
                        tmp = e * (Math.exp(Q1) - Math.exp(-Q1)) / (Math.exp(Q1) + Math.exp(-Q1));
                        Q1 = Q + e * 0.5 * Math.log((1.0 + tmp) / (1.0 - tmp));
                    }
                    x = l;
                    y = Math.atan((Math.exp(Q1) - Math.exp(-Q1)) / 2.0);
                    x += this.m_lambda0;
                    break;
                }
                case EXTENDED: {
                    double lon0 = projectionParams.getOriginLongitude().getValue(AngularUnit.DEGREE) * Math.PI / 180.0;
                    x /= projectionParams.getEllipsoid().getSemiMajorAxis();
                    y /= projectionParams.getEllipsoid().getSemiMajorAxis();
                    if (Math.abs(x / this.m_Qn) > 2.623395162778) {
                        x = x < 0.0 ? -2.623395162778 * this.m_Qn : 2.623395162778 * this.m_Qn;
                    }
                    double Cn = y;
                    double Ce = x;
                    Cn = (Cn - this.m_Zb) / this.m_Qn;
                    double[] vals = new double[2];
                    Cn += TransverseMercatorFactory.clenS(this.m_utg, 2.0 * Cn, 2.0 * (Ce /= this.m_Qn), vals);
                    Ce += vals[1];
                    Ce = 2.0 * (Math.atan(Math.exp(Ce)) - 0.7853981633974483);
                    double sin_Cn = Math.sin(Cn);
                    double cos_Cn = Math.cos(Cn);
                    double sin_Ce = Math.sin(Ce);
                    double cos_Ce = Math.cos(Ce);
                    Ce = Math.atan2(sin_Ce, cos_Ce * cos_Cn);
                    Cn = Math.atan2(sin_Cn * cos_Ce, Math.hypot(sin_Ce, cos_Ce * cos_Cn));
                    y = TransverseMercatorFactory.gatg(this.m_cgb, Cn);
                    x = Ce;
                    x += lon0;
                    x = TransverseMercatorFactory.adjlon(x);
                    break;
                }
                default: {
                    double M = this.m_M0 + y / this.m_k0;
                    double mu = M / this.m_MultY;
                    double mu2 = 2.0 * mu;
                    double cosmu2 = Math.cos(mu2);
                    double phi1 = mu + Math.sin(mu2) * (this.m_Aprime2 + cosmu2 * (this.m_Bprime2 + cosmu2 * (this.m_Cprime2 + cosmu2 * this.m_Dprime2)));
                    double cosphi1 = Math.cos(phi1);
                    if (cosphi1 == 0.0) {
                        x = this.m_lambda0;
                        y = y > 0.0 ? 1.5707963267948966 : -1.5707963267948966;
                        break;
                    }
                    double C1 = this.m_eprime2 * cosphi1 * cosphi1;
                    double C12 = C1 * C1;
                    double tanphi1 = Math.tan(phi1);
                    double T1 = tanphi1 * tanphi1;
                    double T12 = T1 * T1;
                    double sinphi1 = Math.sin(phi1);
                    double temp = 1.0 - projectionParams.getEllipsoid().getEccentricitySquared() * sinphi1 * sinphi1;
                    double N1 = projectionParams.getEllipsoid().getSemiMajorAxis() / Math.sqrt(temp);
                    double R1 = projectionParams.getEllipsoid().getSemiMajorAxis() * (1.0 - projectionParams.getEllipsoid().getEccentricitySquared()) / Math.pow(temp, 1.5);
                    double D = x / (N1 * this.m_k0);
                    double D2 = D * D;
                    double D3 = D * D2;
                    double D4 = D2 * D2;
                    double D5 = D2 * D3;
                    double D6 = D3 * D3;
                    y = phi1 - N1 * tanphi1 / R1 * (D2 / 2.0 - (5.0 + 3.0 * T1 + 10.0 * C1 - 4.0 * C12 - 9.0 * this.m_eprime2) * D4 / 24.0 + (61.0 + 90.0 * T1 + 298.0 * C1 + 45.0 * T12 - 252.0 * this.m_eprime2 - 3.0 * C12) * D6 / 720.0);
                    x = this.m_lambda0 + (D - (1.0 + 2.0 * T1 + C1) * D3 / 6.0 + (5.0 - 2.0 * C1 + 28.0 * T1 - 3.0 * C12 + 8.0 * this.m_eprime2 + 24.0 * T12) * D5 / 120.0) / cosphi1;
                }
            }
            destination.setXY(CoordSysUtilities.convertClip(x, AngularUnit.RADIAN, AngularUnit.DEGREE, -360.0, 360.0), CoordSysUtilities.convertClip(y, AngularUnit.RADIAN, AngularUnit.DEGREE, -90.0, 90.0));
            return destination;
        }

        private DirectPosition ConvertDanish(DirectPosition dp, DanishDirection danishDirection) {
            int InitOfst;
            double[] pCoefficients;
            double x = dp.getX();
            double y = dp.getY();
            switch (danishDirection) {
                case ToDanish: {
                    pCoefficients = this.m_pCoeffUTM2Danish;
                    break;
                }
                case ToUTM: {
                    pCoefficients = this.m_pCoeffDanish2UTM;
                    break;
                }
                default: {
                    throw new UnsupportedOperationException(danishDirection.toString());
                }
            }
            int PolyDegree = (int)pCoefficients[0];
            int offsetY = InitOfst = (PolyDegree + 1) * (PolyDegree + 2) / 2 + 1;
            int offsetX = InitOfst * 2;
            double pYCoefficient = pCoefficients[offsetY];
            double pXCoefficient = pCoefficients[offsetX];
            double InitNorthing = y - pYCoefficient;
            double InitEasting = x - pXCoefficient;
            y = pCoefficients[--offsetY];
            x = pCoefficients[--offsetX];
            for (int outer = PolyDegree; outer > 0; --outer) {
                double WorkNorthing = pCoefficients[--offsetY];
                double WorkEasting = pCoefficients[--offsetX];
                for (int inner = PolyDegree; inner >= outer; --inner) {
                    WorkNorthing = InitNorthing * WorkNorthing + pCoefficients[--offsetY];
                    WorkEasting = InitEasting * WorkEasting + pCoefficients[--offsetX];
                }
                y = InitEasting * y + WorkNorthing;
                x = InitNorthing * x + WorkEasting;
            }
            return new DirectPosition(x, y);
        }

        private static DirectPosition clipPointToRect(DoubleRect rect, double x, double y) {
            x = Math.max(x, rect.getMinX());
            x = Math.min(x, rect.getMaxX());
            y = Math.max(y, rect.getMinY());
            y = Math.min(y, rect.getMaxY());
            return new DirectPosition(x, y);
        }

        static enum DanishDirection {
            ToDanish,
            ToUTM;

        }
    }
}

